<?php

/**
 * Class TurboPartScraperService
 *
 * Cette classe permet de scraper les données depuis un site web sécurisé par un login.
 * Elle gère la connexion à l'application, la récupération de données spécifiques, la sauvegarde des résultats dans des fichiers JSON et l'affichage dans un navigateur.
 */
class TurboPartScraperService {

    private $errors;
    private $jsonContent;
    private array $payload;

    /**
     * TurboPartScraperService constructor.
     *
     * Initialise la classe en prenant les paramètres nécessaires pour se connecter au site, récupérer des données, et sauvegarder les résultats.
     *
     * @param array $list_ids Liste des identifiants de pièces pour lesquelles les données doivent être récupérées.
     * @param bool $uniqueSave Indique si les chaque données doivent être sauvegardées dans un fichier unique. Par défaut : false.
     * @param string $username Le nom d'utilisateur pour se connecter.
     * @param string $password Le mot de passe pour se connecter.
     * @param string $save_files_path Le chemin où les fichiers JSON seront sauvegardés. Par défaut : "saves".
     * @param string $cookie_file Le fichier où les cookies de session seront stockés. Par défaut : "cookies.txt".
     * @param string $login_url L'URL de la page de connexion. Par défaut : "https://www.melett.fr/wp-admin/admin-ajax.php".
     * @param string $target_url L'URL de la page cible contenant les informations de pièces détachées. Par défaut : "https://www.melett.fr/master-pricing/?partcode=".
     * @param array $headers Tableau des en-têtes HTTP à utiliser pour les requêtes. Par défaut, un User-Agent pour simuler un navigateur.
     * @return void
     */
    public function __construct(
        private array $list_ids,
        private bool $uniqueSave = false,
        private string $username,
        private string $password,
        private string $save_files_path = "saves",
        private string $cookie_file = "cookies.txt",
        private string $login_url = "https://www.melett.fr/wp-admin/admin-ajax.php",
        private string $target_url = "https://www.melett.fr/master-pricing/?partcode=",
        private array $headers = [
            "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/114.0.0.0 Safari/537.36"
        ],
    )
    {
        if(!file_exists($this->save_files_path)){
            mkdir($this->save_files_path);
        }

        $this->payload = [
            "username" => $this->username,
            "password" => $this->password,
            "keep_me_logged_in" => "keep_me_logged_in",
            "action" => "melett_customer_login"
        ];
        $this->errors = [];
        $this->jsonContent = [];

        $response = preg_replace('/^\xEF\xBB\xBF/', '',trim($this->make_request($this->login_url, $this->payload, $this->headers)));
        $response_data = json_decode($response, true);

        if ($response_data['success']) {
            $all_results = [];
            foreach ($this->list_ids as $id) {
                $use_target_url = $target_url . $id;
    
                $target_response = $this->make_request($use_target_url,null,null);
    
                $mPDetailsData = $this->extract_mPDetailsData($target_response);
    
                if ($mPDetailsData) {
                    $all_results[] = $mPDetailsData;

                    $imageLinks = $this->extract_image_links($target_response);

                    if ($imageLinks) {
                        foreach ($imageLinks as $link) {

                            if($this->uniqueSave){
                                $imageDir = $this->save_files_path . '/' . $id . '/images';

                                if(!file_exists($imageDir)){
                                    mkdir($imageDir);
                                }
                                $filename = str_replace('https://www.melett.fr/wp-content/themes/meletttheme/media/product-images/', '', $link);
                                $localPath = $imageDir . '/' . $filename;

                                file_put_contents($localPath, file_get_contents($link));
                            }

                            if($this->uniqueSave){
                                $all_results['image_link'][] = $imageDir . '/' . $filename;
                            }else{
                                $all_results['image_link'][] = $link;
                            }
                        }
                    }else{
                        $all_results[] = ['image_link' => []];
                    }

                    if($this->uniqueSave){
                        if(!file_exists($this->save_files_path . '/' . $id)){
                            mkdir($this->save_files_path . '/' . $id);
                        }
                        $mPDetailsData = json_encode($mPDetailsData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                        file_put_contents($this->save_files_path . '/' . $id . "/" . date("Y-m-d_H-i-s_") . round(microtime(true) * 1000) . ".json", $mPDetailsData);
                    }
                } else {
                    $error = "Impossible de trouver les données mPDetailsData pour l'ID: $id";
                    $this->errors[] = $error;

                    if($this->uniqueSave){
                        if(!file_exists($this->save_files_path . '/' . $id)){
                            mkdir($this->save_files_path . '/' . $id);
                        }
                        $error = json_encode($error, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                        file_put_contents($this->save_files_path . '/' . $id . "/" . date("Y-m-d_H-i-s_") . round(microtime(true) * 1000) . ".json", $error);
                    }
                }
            }

            $this->jsonContent['data'] = $all_results;
            $this->jsonContent['errors'] = $this->errors;

            $this->jsonContent = json_encode($this->jsonContent, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

            if(!file_exists($this->save_files_path . '/results')){
                mkdir($this->save_files_path . '/results');
            }

            file_put_contents($this->save_files_path . "/results/" . date("Y-m-d_H-i-s_") . round(microtime(true) * 1000) . ".json", $this->jsonContent);

            echo $this->jsonContent;
    
        } else {
            $this->errors[] = "Échec de la connexion : Utilisateur ou mot de passe incorrect";

            $this->jsonContent['errors'] = $this->errors;

            $this->jsonContent = json_encode($this->jsonContent, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

            if(!file_exists($this->save_files_path . '/results')){
                mkdir($this->save_files_path . '/results');
            }

            file_put_contents($this->save_files_path . "/results/" . date("Y-m-d_H-i-s_") . round(microtime(true) * 1000) . ".json", $this->jsonContent);

            echo $this->jsonContent;
        }
    }

    /**
     * Effectue une requête HTTP et retourne la réponse.
     *
     * Cette méthode utilise cURL pour effectuer une requête GET ou POST en fonction des paramètres fournis. 
     * Elle gère les cookies de session pour maintenir la connexion pendant les requêtes successives.
     *
     * @param string $url L'URL à requêter.
     * @param array|null $data Données POST à envoyer. Si null, une requête GET est effectuée.
     * @param array|null $headers En-têtes HTTP supplémentaires à envoyer avec la requête. Si null, les en-têtes par défaut sont utilisés.
     * @return string La réponse de la requête.
     */
    private function make_request($url, $data = null, $headers = []) {
        $ch = curl_init();
    
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    
        if ($data !== null) {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        }
    
        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
    
        curl_setopt($ch, CURLOPT_COOKIEJAR, $this->cookie_file);
        curl_setopt($ch, CURLOPT_COOKIEFILE, $this->cookie_file);
    
        $response = curl_exec($ch);
    
        curl_close($ch);
        return $response;
    }
    
    /**
     * Extrait les données de mPDetailsData depuis la page HTML.
     *
     * Cette méthode recherche dans le contenu HTML de la page les données sous forme de variable JavaScript et les retourne sous forme de tableau.
     *
     * @param string $html Le contenu HTML de la page cible.
     * @return array|null Les données extraites sous forme de tableau ou null si les données n'ont pas été trouvées.
     */
    public function extract_mPDetailsData($html) {
        $pattern = '/var mPDetailsData = (\{.*?\});/s';
        preg_match($pattern, $html, $matches);
    
        if (isset($matches[1])) {
            return json_decode($matches[1], true);
        }
    
        return null;
    }

    /**
     * Extrait les liens des images Melett depuis la page HTML.
     *
     * @param string $html Le contenu HTML de la page cible.
     * @return array|null Les URLs des images extraites ou null si aucune image n'a été trouvée.
     */
    public function extract_image_links($html) {
        $pattern = '/<img src="(https:\/\/www\.melett\.fr\/wp-content\/themes\/meletttheme\/media\/product-images\/[^"]+)"[^>]*>/';
        preg_match_all($pattern, $html, $matches);

        return isset($matches[1]) ? array_unique($matches[1]) : null;
    }
}

?>
